// $Id: CFileInformation.cpp,v 1.8 2007/02/08 21:06:44 paul Exp $

/*
 * All contents of this source code are copyright 2005 Exp Digital Uk.
 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy
 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
 * All content is the Intellectual property of Exp Digital Uk.
 * Certain sections of this code may come from other sources. They are credited where applicable.
 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
 */

#include "CFileInformation.hpp"
using Exponent::Host::CFileInformation;

#ifdef WIN32
#pragma warning (disable : 4307)	// warning C4307: '+' : integral constant overflow
#endif

//	===========================================================================
EXPONENT_CLASS_IMPLEMENTATION(CFileInformation, CCountedObject);

//	===========================================================================
CFileInformation::CFileInformation() : m_attribute(e_unknown) , m_fileSizeInBytes(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileInformation);
}

//	===========================================================================
CFileInformation::CFileInformation(const CFileInformation &info) : m_attribute(e_unknown) , m_fileSizeInBytes(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileInformation);
	m_attribute		  = info.m_attribute;
	m_creationTime	  = info.m_creationTime;
	m_accessTime	  = info.m_accessTime;
	m_writeTime		  = info.m_writeTime;
	m_fileSizeInBytes = info.m_fileSizeInBytes;
	m_filename		  = info.m_filename;
	m_shortFilename	  = info.m_shortFilename;
	m_foldername	  = info.m_foldername;
	m_fullPath		  = info.m_fullPath;
}

#ifdef WIN32
//	===========================================================================
CFileInformation::CFileInformation(WIN32_FIND_DATA &fileInfo) : m_attribute(e_unknown) , m_fileSizeInBytes(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileInformation);
	this->setFileInformationFromData(fileInfo);
}
#else
//	===========================================================================
CFileInformation::CFileInformation(dirent *directoryEntry) : m_attribute(e_unknown) , m_fileSizeInBytes(0)
{
	EXPONENT_CLASS_CONSTRUCTION(CFileInformation);
	this->setFilename(CString(directoryEntry->d_name));
	m_shortFilename   = m_filename;
	m_fileSizeInBytes = (double)directoryEntry->d_reclen;
	switch(directoryEntry->d_type)
	{
		case DT_UNKNOWN:	m_attribute = e_unknown;	break;
		case DT_DIR:		m_attribute = e_directory;	break;
		case DT_CHR:
		case DT_BLK:
		case DT_REG:		m_attribute = e_normal;		break;
		case DT_FIFO:
		case DT_LNK:
		case DT_SOCK:
		case DT_WHT:		m_attribute = e_system;		break;
	}
}
#endif

//	===========================================================================
CFileInformation::~CFileInformation()
{
	EXPONENT_CLASS_DESTRUCTION(CFileInformation);
}

//	===========================================================================
CFileInformation &CFileInformation::operator = (const CFileInformation &info)
{
	if (&info != this)
	{
		m_attribute		  = info.m_attribute;
		m_creationTime	  = info.m_creationTime;
		m_accessTime	  = info.m_accessTime;
		m_writeTime		  = info.m_writeTime;
		m_fileSizeInBytes = info.m_fileSizeInBytes;
		m_filename		  = info.m_filename;
		m_shortFilename	  = info.m_shortFilename;
		m_foldername	  = info.m_foldername;
		m_fullPath		  = info.m_fullPath;
	}

	return *this;
}

#ifdef WIN32
//	===========================================================================
void CFileInformation::setFileInformationFromData(WIN32_FIND_DATA &fileInfo)
{
	m_creationTime.setTime(fileInfo.ftCreationTime);
	m_accessTime.setTime(fileInfo.ftLastAccessTime);
	m_writeTime.setTime(fileInfo.ftLastWriteTime);
	m_fileSizeInBytes = (fileInfo.nFileSizeHigh * (MAXDWORD + 1)) + fileInfo.nFileSizeLow;
	m_filename = fileInfo.cFileName;
	m_shortFilename = fileInfo.cAlternateFileName;

	// To enable cross platform compatiblity, we only check for a single attribute (eeeeek)
	// TODO - Enable multiple attributes, rather than a single attribute

	// On mac the supported attributes are:
	// Unknown file attribute
	// Directory attribute
	// Normal (file) attribute
	// System attribute
	if ((fileInfo.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) > 0)
	{
		m_attribute = e_directory;
	}
	else if ((fileInfo.dwFileAttributes & FILE_ATTRIBUTE_NORMAL) > 0)
	{
		m_attribute = e_normal;
	}
	else if ((fileInfo.dwFileAttributes & FILE_ATTRIBUTE_SYSTEM) > 0)
	{
		m_attribute = e_system;
	}
	else
	{
		m_attribute = e_normal;
	}

	/*
	switch(fileInfo.dwFileAttributes)
	{
		case FILE_ATTRIBUTE_ARCHIVE:		m_attribute = e_archive;			break;
		case FILE_ATTRIBUTE_COMPRESSED:		m_attribute = e_compressed;			break;
		case FILE_ATTRIBUTE_DIRECTORY:		m_attribute = e_directory;			break;
		case FILE_ATTRIBUTE_ENCRYPTED:		m_attribute = e_encryted;			break;
		case FILE_ATTRIBUTE_HIDDEN:			m_attribute = e_hidden;				break;
		case FILE_ATTRIBUTE_NORMAL:			m_attribute = e_normal;				break;
		case FILE_ATTRIBUTE_OFFLINE:		m_attribute = e_offline;			break;
		case FILE_ATTRIBUTE_READONLY:		m_attribute = e_readOnly;			break;
		case FILE_ATTRIBUTE_REPARSE_POINT:	m_attribute = e_reparse;			break;
		case FILE_ATTRIBUTE_SYSTEM:			m_attribute = e_system;				break;
		case FILE_ATTRIBUTE_TEMPORARY:		m_attribute = e_temporary;			break;
		default:							m_attribute = e_normal;				break;
	}
	*/
}
#endif

//	===========================================================================
void CFileInformation::setFileAttribute(const EFileAttribute attribute)
{
	m_attribute = attribute;
}

//	===========================================================================
void CFileInformation::setCreationTime(const CTime &creationTime)
{
	m_creationTime = creationTime;
}

//	===========================================================================
void CFileInformation::setLastAccessTime(const CTime &lastAccess)
{
	m_accessTime = lastAccess;
}

//	===========================================================================
void CFileInformation::setLastWriteTime(const CTime &lastWrite)
{
	m_writeTime = lastWrite;
}

//	===========================================================================
void CFileInformation::setSizeInBytes(const double size)
{
	m_fileSizeInBytes = size;
}

//	===========================================================================
void CFileInformation::setFilename(const CString &filename)
{
	m_filename = filename;
}

//	===========================================================================
void CFileInformation::setFoldername(const CString &foldername)
{
	m_foldername = foldername;
	m_fullPath   = foldername;
#ifndef WIN32
	m_fullPath.appendSystemSeperator();		// Bloody macs :@
#endif
	m_fullPath.appendString(m_filename);

}

//	===========================================================================
int CFileInformation::compareFiles(const CFileInformation **info1, const CFileInformation **info2)
{
	if (*info1 == *info2)
	{
		return 0;
	}
	else if (*info1 == NULL)
	{
		return 1;
	}
	else if (*info2 == NULL)
	{
		return -1;
	}
	else
	{
		// Cast them over
		CFileInformation *theInfo1 = (CFileInformation*)(*info1);
		CFileInformation *theInfo2 = (CFileInformation*)(*info2);

		if (theInfo1->isDirectory() && !theInfo2->isDirectory())
		{
			return -1;
		}
		if (!theInfo1->isDirectory() && theInfo2->isDirectory())
		{
			return 1;
		}

		return strcmp(theInfo1->getFullPath().getString(), theInfo2->getFullPath().getString());
		//return strcmp(((CFileInformation *)(*class1))->getFullPath().getString(), ((CFileInformation *)(*class2))->getFullPath().getString());
	}
}